<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Dtos\Common;

use Erlage\Photogram\Tools\Fso\Storage;
use Erlage\Photogram\Data\Dtos\AbstractDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintValidatorDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintTransformerDTO;

class DisplayItemDTO extends AbstractDTO
{
    /**
     * to comply with DTO interface
     */
    use TraitComplaintValidatorDTO;
    use TraitComplaintTransformerDTO;

    /*
    |--------------------------------------------------------------------------
    | key map
    |--------------------------------------------------------------------------
    */

    const KEY_HOST = 'host';

    const KEY_TYPE = 'type';

    const KEY_FILESPACE = 'filespace';

    const KEY_IDENTIFIER = 'identifier';

    const KEY_URL_ORIGINAL = 'url_original';

    const KEY_URL_COMPRESSED = 'url_compressed';

    /*
    |--------------------------------------------------------------------------
    | available types
    |--------------------------------------------------------------------------
    */

    const TYPE_IMAGE = 'image';

    const TYPE_VIDEO = 'video';

    /*
    |--------------------------------------------------------------------------
    | available namespaces
    |--------------------------------------------------------------------------
    */

    const FILESPACE_USER = 'user';

    const FILESPACE_POST = 'post';

    const FILESPACE_COLLECTION = 'collection';

    const FILESPACE_PLACEHOLDER = 'placeholder';

    /*
    |--------------------------------------------------------------------------
    | data
    |--------------------------------------------------------------------------
    */

    /**
     * @var string
     */
    private $host;

    /**
     * @var string
     */
    private $type;

    /**
     * @var string
     */
    private $filespace;

    /**
     * @var string
     */
    private $identifier;

    /*
    |--------------------------------------------------------------------------
    | getters
    |--------------------------------------------------------------------------
    */

    public function isImage(): bool
    {
        return $this -> type == self::TYPE_IMAGE;
    }

    public function isVideo(): bool
    {
        return $this -> type == self::TYPE_VIDEO;
    }

    public function getHost(): string
    {
        return $this -> host;
    }

    public function getType(): string
    {
        return $this -> type;
    }

    public function getFilespace(): string
    {
        return $this -> filespace;
    }

    public function getIdentifier(): string
    {
        return $this -> identifier;
    }

    /*
    |--------------------------------------------------------------------------
    | object ids
    |--------------------------------------------------------------------------
    */

    public function getOriginalObjectId(): string
    {
        $path = $this -> getFilespace() . '/';
        $path .= $this -> getType() . '/';
        $path .= $this -> getIdentifier();

        return $path;
    }

    public function getCompressedObjectId(): string
    {
        $path = $this -> getFilespace() . '/';
        $path .= $this -> getType() . '/';
        $path .= 'c_' . $this -> getIdentifier();

        return $path;
    }

    /*
    |--------------------------------------------------------------------------
    | setters
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public function setHost(string $host)
    {
        $this -> host = $host;

        return $this;
    }

    /**
     * @return static 
     */
    public function setIdentifier(string $identifier)
    {
        $this -> identifier = $identifier;

        return $this;
    }

    /**
     * @return static 
     */
    public function setFilespace(string $filespace)
    {
        $this -> filespace = $filespace;

        return $this;
    }

    /**
     * @return static 
     */
    public function setType(string $type)
    {
        $this -> type = $type;

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | from json
    |--------------------------------------------------------------------------
    */

    public static function fromJsonMap(array $data): self
    {
        return (new static())
            -> setHost((string) $data[self::KEY_HOST])
            -> setType((string) $data[self::KEY_TYPE])
            -> setFilespace((string) $data[self::KEY_FILESPACE])
            -> setIdentifier((string) $data[self::KEY_IDENTIFIER]);
    }

    /*
    |--------------------------------------------------------------------------
    | serialization for json_encode
    |--------------------------------------------------------------------------
    */

    public function jsonSerialize()
    {
        return array(
            self::KEY_HOST           => $this -> host,
            self::KEY_TYPE           => $this -> type,
            self::KEY_FILESPACE      => $this -> filespace,
            self::KEY_IDENTIFIER     => $this -> identifier,

            self::KEY_URL_ORIGINAL   => Storage::disk($this -> host)
                -> setObjectId($this -> getOriginalObjectId())
                -> getPublicUrl(),

            self::KEY_URL_COMPRESSED => Storage::disk($this -> host)
                -> setObjectId($this -> getCompressedObjectId())
                -> getPublicUrl(),
        );
    }

    /*
    |--------------------------------------------------------------------------
    | implement DTO methods
    |--------------------------------------------------------------------------
    */

    /**
     * @param DisplayItemDTO $displayItemDTO
     */
    public static function serialize($displayItemDTO): string
    {
        return \json_encode(
             array(
                self::KEY_HOST           => $displayItemDTO -> host,
                self::KEY_TYPE           => $displayItemDTO -> type,
                self::KEY_FILESPACE      => $displayItemDTO -> filespace,
                self::KEY_IDENTIFIER     => $displayItemDTO -> identifier,
            )
        );
    }

    /**
     * @return DisplayItemDTO
     */
    public static function deSerialize(string $displayItemDTO)
    {
        $data = \json_decode($displayItemDTO, true);

        return static::fromJsonMap($data);
    }
}
